<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Illuminate\Support\Carbon;
use App\Http\Traits\Bmf;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use App\Http\Traits\HttpResponses;
use Illuminate\Support\Facades\Mail;
use App\Models\Contests;
use App\Models\UserCart;
use App\Models\Orders;
use App\Models\OrdersItems;
use App\Models\PaymentHistory;
use App\Models\OrdersTickets;
use Illuminate\Support\Facades\Storage;
use App\Services\EmailService;


use Sebdesign\VivaPayments\Enums\TransactionStatus;
use Sebdesign\VivaPayments\Facades\Viva;
use Sebdesign\VivaPayments\Requests\CreatePaymentOrder;
use Sebdesign\VivaPayments\Requests\Customer;
use Sebdesign\VivaPayments\VivaException;
use Illuminate\Http\RedirectResponse;

class PaymentController extends Controller
{
    use Bmf, HttpResponses;

    protected $user;
    protected $cart;
    protected $contest;
    protected $orders;
    protected $orderItems;
    protected $paymentHistory;
    protected $orderTickets;
    protected $emailService;

    public function __construct(
        User $_user,
        UserCart $_cart,
        Contests $_contest,
        Orders  $_orders,
        OrdersItems  $_ordersItems,
        PaymentHistory  $_paymentHistory,
        OrdersTickets $_ordersTickets,
        EmailService $emailService
    ) {
        $this->user             =   $_user;
        $this->cart             =   $_cart;
        $this->contest          =   $_contest;
        $this->orders           =   $_orders;
        $this->orderItems       =   $_ordersItems;
        $this->paymentHistory   =   $_paymentHistory;
        $this->orderTickets     =   $_ordersTickets;
        $this->emailService     =   $emailService;
    }

    public function checkout(Request $request): RedirectResponse
    {
        try {
            $user = auth()->user();
            $data   =   $request->all();
            $lineItems = [];
            $totalPrice = 0;
            $cartItems = $data['item'];
            $customerTrns = null;
            $totalQty = 0;
            foreach ($cartItems as $key => $cartItem) {
                $item = $this->contest->find($cartItem['id']);
                $itemImage = $item->images[0]->path ?? null;
                if (!empty($itemImage) && Storage::exists($itemImage)) {
                    $image = asset(Storage::url($itemImage));
                } else {
                    $image = asset('frontend/assets/images/contest/s1.png');
                }

                $lineItems[] = [
                    'price_data' => [
                        'currency' => 'EUR',
                        'product_data' => [
                            'name' => $item->title,
                            'images' => [$image]
                        ],
                        'unit_amount' => $cartItem['price'] * 100,
                    ],
                    'quantity' => $cartItem['quantity'],
                ];
                $totalQty = $cartItem['quantity'];
                $customerTrns .= $item->title . ' - Tickets : ' . $totalQty . ($key + 1 < count($cartItems) ? ' | ' : '');
                $totalPrice += ($cartItem['price'] * $cartItem['quantity']);
            }

            $orderCode = Viva::orders()->create(new CreatePaymentOrder(
                amount: $totalPrice * 100,
                customerTrns: $customerTrns ?? 'Total Items ' . count($cartItems),
                customer: new Customer(
                    email: $user->email,
                    fullName: $user->name,
                    countryCode: 'GB',
                    requestLang: 'en-GB',
                ),
                sourceCode: config('services.viva.sourceCode')
            ));
        } catch (VivaException $e) {
            report($e);
            return back()->withErrors($e->getMessage());
        }


        if ($orderCode) {
            $response = DB::transaction(function () use ($data, $orderCode, $cartItems, $totalPrice, $customerTrns) {
                // Save all related data in db.
                $order = $this->orders->create([
                    'user_id'       =>  $data['user'],
                    'oid'           =>  time(),
                    'name'          =>  $data['name'],
                    'email'         =>  $data['email'],
                    'mobile'        =>  $data['mobile'] ?? 0,
                    'discount'      =>  0,
                    'tax'           =>  0,
                    'grand_total'   =>  $totalPrice,
                    'session_id'    =>  $orderCode,
                    'order_response' =>  !empty($customerTrns) ? $customerTrns : 'Total Items ' . count($cartItems)
                ]);
                if ($order) {
                    foreach ($cartItems as $cartItem) {
                        $item = $this->contest->find($cartItem['id']);
                        $total = $cartItem['price'] * $cartItem['quantity'];
                        $this->orderItems->create([
                            'order_id'      =>  $order->id,
                            'contest_id'    =>  $item->id,
                            'quantity'      =>  $cartItem['quantity'],
                            'price'         =>  $cartItem['price'],
                            'item_discount' =>  0,
                            'item_tax'      =>  0,
                            'total'         =>  $total
                        ]);
                    }

                    return true;
                }

                return false;
            });

            if ($response) {
                $redirectUrl = Viva::orders()->redirectUrl(
                    ref: $orderCode,
                    color: '0000ff',
                    paymentMethod: 23,
                );

                return redirect()->away($redirectUrl);
            }
        }
    }


    /**
     * Redirect from the checkout page and get the order details from the API.
     */
    public function confirm(Request $request): RedirectResponse
    {
        try {
            $transaction = Viva::transactions()->retrieve($request->input('t'));
            dd($transaction);
        } catch (VivaException $e) {
            report($e);
            return back()->withErrors($e->getMessage());
        }

        dd($transaction->statusId);

        // $status = match ($transaction->statusId) {
        //     case TransactionStatus::PaymentPending: 'The order is pending.',
        //     case TransactionStatus::PaymentSuccessful: 'The order is paid.',
        //     case TransactionStatus::Error: 'The order was not paid.',
        // }

        // return view('order/success', compact('status'));
    }

    public function success(Request $request)
    {
        try {
            $transaction = Viva::transactions()->retrieve($request->input('t'));
            $transactionId = $request->input('t');
            $sessionId = $transaction?->orderCode;
            $order = $this->orders->where('session_id', $sessionId)->first();
            if (!$order) {
                throw new NotFoundHttpException();
            }

            $user = $order->user;

            if ($transaction->statusId == TransactionStatus::PaymentSuccessful) {
                $status =   'complete';
                $paymentStatus =   'paid';
            }

            if ($transaction->statusId == TransactionStatus::PaymentPending) {
                $status =   'pending';
                $paymentStatus = 'processing';
            }

            $ip_address = $request->ip();
            $user = $order->user;
            if ($user && $order->payment_status === 'unpaid') {
                $order->status              =   $status;
                $order->payment_status      =   $paymentStatus;
                $order->payment_response    =   json_encode($transaction);
                $order->transaction_id      =   $transactionId;
                if ($order->save()) {
                    //Generate Ticket Numbers
                    foreach ($order->items as $key => $item) {
                        $contest = $this->contest->find($item->contest_id);
                        if ($contest) {
                            for ($i = 0; $i < $item->quantity; $i++) {
                                $checkTickets = $this->orderTickets->where(['contest_id' => $item->contest_id])
                                    ->get()->count();

                                $ticketNumber = sprintf('%03s', $checkTickets + 1);
                                $ticketId = strtoupper($item->contest->contest_no) . '-' . $ticketNumber;
                                $ticket = $this->orderTickets->where(['contest_id' => $item->contest_id])
                                    ->where('ticket_id', $ticketNumber)->where('item_id', $item->id)->first();
                                // Log::info($checkTickets . ' - ' . print_r($ticketNumber, true));

                                if (!$ticket) {
                                    $this->orderTickets->create([
                                        'user_id'   =>  $item->order->user_id,
                                        'order_id'  =>  $item->order->id,
                                        'contest_id' =>  $contest->id,
                                        'item_id'   =>  $item->id,
                                        'ticket_id' =>  $ticketId,
                                        'ticket_no' =>  $ticketNumber,
                                    ]);
                                }
                            }
                        }
                    }

                    $lastHistory = $this->paymentHistory->where('order_id', $order->id)->first();
                    $checkHistory = $this->paymentHistory->where('order_id', $order->id)
                        ->where('status', $status)
                        ->first();
                    if (!$checkHistory) {
                        $this->paymentHistory->create([
                            'order_id'  =>  $order->id,
                            'user_id'   =>  $order->user_id,
                            'discount'  =>  $order->discount,
                            'tax'       =>  $order->tax,
                            'grand_total'   =>  $order->grand_total,
                            'status'        =>  $status,
                            'payment_status' =>  $paymentStatus,
                            'previous_response' =>  !empty($lastHistory) ? $lastHistory->current_response : json_encode($transaction),
                            'current_response'  =>  json_encode($transaction)
                        ]);
                    }

                    // Delete cart items
                    $this->cart->where(function ($query) use ($order, $ip_address) {
                        $query->where('user_id', $order->user_id)
                            ->orWhere('ip_address', $ip_address);
                    })->delete();

                    //Send Mail
                    $this->sendTicketPurchaseDetailsMail($user, $order, $transaction);
                }
            }

            return redirect()->route('frontend.user.checkout.success.details', ['order' => $order->id, 'session' => $transactionId]);
        } catch (VivaException $e) {
            report($e);
            return redirect()->route('frontend.user.checkout.cart')->withErrors($e->getMessage());
        }
    }

    public function successDetails(Request $request, Orders $order, $session)
    {
        $responseData = json_decode($order?->payment_response);
        $sessionId = $session ?? $request->session;
        $session = Viva::transactions()->retrieve($sessionId);
        return view('application.checkout.success', compact('order', 'session'));
    }

    public function cancel(Request $request)
    {
        try {
            $transaction = Viva::transactions()->retrieve($request->input('t'));
            $transactionId = $request->input('t');
            $sessionId = $transaction?->orderCode;
            $order = $this->orders->where('session_id', $sessionId)->first();
            if (!$order) {
                throw new NotFoundHttpException();
            }

            $user = $order->user;
            // $transaction->statusId == TransactionStatus::Error
            $status =   'cancel';
            $paymentStatus =   'failed';

            $user = $order->user;
            if ($user && $order->payment_status === 'unpaid') {
                $order->status              =   $status;
                $order->payment_status      =   $paymentStatus;
                $order->payment_response    =   json_encode($transaction);
                $order->transaction_id      =   $transactionId;
                if ($order->save()) {
                    $this->orderTickets->where('order_id', $order->id)->delete();
                    $lastHistory = $this->paymentHistory->where('order_id', $order->id)->first();
                    $checkHistory = $this->paymentHistory->where('order_id', $order->id)
                        ->where('status', $status)
                        ->first();
                    if (!$checkHistory) {
                        $this->paymentHistory->create([
                            'order_id'  =>  $order->id,
                            'user_id'   =>  $order->user_id,
                            'discount'  =>  $order->discount,
                            'tax'       =>  $order->tax,
                            'grand_total'   =>  $order->grand_total,
                            'status'        =>  $status,
                            'payment_status' =>  $paymentStatus,
                            'previous_response' =>  !empty($lastHistory) ? $lastHistory->current_response : json_encode($transaction),
                            'current_response'  =>  json_encode($transaction)
                        ]);
                    }
                }
            }

            return view('application.checkout.failed');

            // return redirect()->route('frontend.user.checkout.success.details', ['order' => $order->id, 'session' => $transactionId]);
        } catch (VivaException $e) {
            report($e);
            return redirect()->route('frontend.user.checkout.cart')->withErrors($e->getMessage());
        }
    }



    // Stripe
    public function checkoutOld(Request $request)
    {
        try {
            //code...

            $data   =   $request->all();
            \Stripe\Stripe::setApiKey(config('stripe.stripe_secret_key'));
            $lineItems = [];
            $totalPrice = 0;
            $cartItems = $data['item'];
            foreach ($cartItems as $cartItem) {
                $item = $this->contest->find($cartItem['id']);
                $itemImage = $item->images[0]->path ?? null;
                if (!empty($itemImage) && Storage::exists($itemImage)) {
                    $image = asset(Storage::url($itemImage));
                } else {
                    $image = asset('frontend/assets/images/contest/s1.png');
                }

                $lineItems[] = [
                    'price_data' => [
                        'currency' => 'EUR',
                        'product_data' => [
                            'name' => $item->title,
                            'images' => [$image]
                        ],
                        'unit_amount' => $cartItem['price'] * 100,
                    ],
                    'quantity' => $cartItem['quantity'],
                ];

                $totalPrice += ($cartItem['price'] * $cartItem['quantity']);
            }

            if (count($lineItems) < 0) {
                return redirect()->back()->with('failed', 'Invalid items to checkout.');
            }

            $session = \Stripe\Checkout\Session::create([
                'line_items' => $lineItems,
                'mode' => 'payment',
                'success_url' => route('frontend.user.checkout.success', [], true) . "?session_id={CHECKOUT_SESSION_ID}",
                'cancel_url' => route('frontend.user.checkout.cancel', [], true) . "?session_id={CHECKOUT_SESSION_ID}",
            ]);

            $response = DB::transaction(function () use ($data, $session, $cartItems, $totalPrice) {
                // Save all related data in db.
                $order = $this->orders->create([
                    'user_id'       =>  $data['user'],
                    'oid'           =>  time(),
                    'name'          =>  $data['name'],
                    'email'         =>  $data['email'],
                    'mobile'        =>  $data['mobile'] ?? 0,
                    'discount'      =>  0,
                    'tax'           =>  0,
                    'grand_total'   =>  $totalPrice,
                    'session_id'    =>  $session->id
                ]);
                if ($order) {
                    foreach ($cartItems as $cartItem) {
                        $item = $this->contest->find($cartItem['id']);
                        $total = $cartItem['price'] * $cartItem['quantity'];
                        $this->orderItems->create([
                            'order_id'      =>  $order->id,
                            'contest_id'    =>  $item->id,
                            'quantity'      =>  $cartItem['quantity'],
                            'price'         =>  $cartItem['price'],
                            'item_discount' =>  0,
                            'item_tax'      =>  0,
                            'total'         =>  $total
                        ]);
                    }

                    return true;
                }

                return false;
            });

            if ($response) {
                return redirect($session->url);
            }

            return redirect()->back()->with('failed', 'Invalid request to processed payment process.');
        } catch (\Throwable $th) {
            return redirect()->back()->with('failed', $th->getMessage());
        }
    }

    public function success_(Request $request)
    {
        \Stripe\Stripe::setApiKey(config('stripe.stripe_secret_key'));
        $sessionId = $request->get('session_id');

        try {
            $session = \Stripe\Checkout\Session::retrieve($sessionId);
            if (!$session) {
                throw new NotFoundHttpException;
            }
            // $customer = \Stripe\Customer::retrieve($session->customer);
            $order = $this->orders->where('session_id', $session->id)->first();
            if (!$order) {
                throw new NotFoundHttpException();
            }

            $user = $order->user;
            if ($order->payment_status === 'unpaid') {
                $order->status              =   'complete';
                $order->payment_status      =   'paid';
                $order->payment_response    =   json_encode($session);
                if ($order->save()) {
                    //Generate Ticket Numbers
                    foreach ($order->items as $key => $item) {
                        $contest = $this->contest->find($item->contest_id);
                        if ($contest) {
                            for ($i = 0; $i < $item->quantity; $i++) {
                                $checkTickets = $this->orderTickets->where(['contest_id' => $contest->id])->get()->count();
                                $ticketNumber = sprintf('%03s', $checkTickets + 1);
                                $ticketId = strtoupper($item->contest->contest_no) . '-' . $ticketNumber;
                                $this->orderTickets->create([
                                    'user_id'   =>  $item->order->user_id,
                                    'order_id'  =>  $item->order->id,
                                    'contest_id' =>  $contest->id,
                                    'item_id'   =>  $item->id,
                                    'ticket_id' =>  $ticketId,
                                    'ticket_no' =>  $ticketNumber,
                                ]);
                            }
                        }
                    }

                    $lastHistory = $this->paymentHistory->where('order_id', $order->id)->first();
                    $checkHistory = $this->paymentHistory->where('order_id', $order->id)
                        ->where('status', $session->status)
                        ->first();
                    if (!$checkHistory) {
                        $this->paymentHistory->create([
                            'order_id'  =>  $order->id,
                            'user_id'   =>  $order->user_id,
                            'discount'  =>  $order->discount,
                            'tax'       =>  $order->tax,
                            'grand_total'   =>  $order->grand_total,
                            'status'        =>  $session->status,
                            'payment_status' =>  $session->status,
                            'previous_response' =>  !empty($lastHistory) ? $lastHistory->current_response : json_encode($session),
                            'current_response'  =>  json_encode($session)
                        ]);
                    }

                    //Send Mail
                    $this->sendTicketPurchaseDetailsMail($user, $order, $session);
                }

                // Delete cart items
                $this->cart->where('user_id', $order->user_id)->delete();
            }

            return redirect()->route('frontend.user.checkout.success.details', ['order' => $order->id, 'session' => $session->id]);
        } catch (\Exception $e) {
            Log::error($e);
            throw new NotFoundHttpException();
        }
    }

    public function successDetails_(Request $request, Orders $order, $session)
    {
        \Stripe\Stripe::setApiKey(config('stripe.stripe_secret_key'));
        $sessionId = $session ?? $request->session;
        $session = \Stripe\Checkout\Session::retrieve($sessionId);
        return view('application.checkout.success', compact('order', 'session'));
    }

    private function sendTicketPurchaseDetailsMail($user, $order, $session)
    {
        if (isset($user->email)) {
            $to = $user->email;
            $subject =  'Your Ticket Purchase Details';
            $view = 'emails.ticket-purchase-details';
            $data['order']      = $order;
            $data['user']       = $user;
            $data['session']    = $session;
            $send = $this->emailService->sendEmail($to, $data, $view, $subject);
            Log::info($send);
            // return view('emails.ticket-purchase-details', ['data' => $data]);
            return $send;
        }
    }

    public function cancel_(Request $request)
    {
        \Stripe\Stripe::setApiKey(config('stripe.stripe_secret_key'));
        $sessionId = $request->get('session_id');
        try {
            $session = \Stripe\Checkout\Session::retrieve($sessionId);
            if (!$session) {
                throw new NotFoundHttpException;
            }
            // $customer = \Stripe\Customer::retrieve($session->customer);
            $order = $this->orders->where('session_id', $session->id)->first();
            if (!$order) {
                throw new NotFoundHttpException();
            }
            if ($order->payment_status === 'unpaid') {
                $order->status              =   'cancel';
                $order->payment_status      =   'failed';
                $order->payment_response    =   json_encode($session);
                $order->save();
            }
            return view('application.checkout.failed');
        } catch (\Exception $e) {
            Log::error($e);
            throw new NotFoundHttpException();
        }
    }

    public function webhook()
    {
        // This is your Stripe CLI webhook secret for testing your endpoint locally.
        $endpoint_secret = env('STRIPE_WEBHOOK_SECRET');

        $payload = @file_get_contents('php://input');
        $sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'];
        $event = null;

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sig_header,
                $endpoint_secret
            );
        } catch (\UnexpectedValueException $e) {
            // Invalid payload
            return response('', 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            return response('', 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'checkout.session.completed':
                $session = $event->data->object;

                $order = Orders::where('session_id', $session->id)->first();
                if ($order && $order->status === 'unpaid') {
                    $order->status = 'paid';
                    $order->save();
                    // Send email to customer
                }

                // ... handle other event types
            default:
                echo 'Received unknown event type ' . $event->type;
        }

        return response('');
    }
}
