<?php

namespace App\Http\Controllers\Application;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ContactPage;
use App\Models\ContactUs;
use App\Models\Orders;
use App\Http\Traits\Bmf as BMF;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Models\AboutUs;
use App\Mail\ContactRequestEmail;
use Illuminate\Support\Facades\Mail;
use App\Models\Contests;
use App\Models\User;
use App\Models\HomePage;
use App\Models\UserCart;
use App\Models\FAQCategoriesModel as FAQCategories;
use App\Models\UsersFavContests;
use App\Models\Subscribers;
use App\Http\Traits\HttpResponses;
use App\Models\OrdersTickets;
use App\Models\Questions;
use App\Models\UserAnswers;
use Carbon\Carbon;
use App\Services\EmailService;

class HomeController extends Controller
{
    use BMF, HttpResponses;

    protected $aboutUs;
    protected $contactUs;
    protected $contactPage;
    protected $homepage;
    protected $contests;
    protected $cart;
    protected $subscribers;
    protected $fav;
    protected $ordersTickets;
    protected $questions;
    protected $userQuestion;
    protected $emailService;

    public function __construct(
        ContactUs $_contact_us,
        ContactPage $_contact_page,
        AboutUs $_about_us,
        Contests $_contests,
        HomePage $_homepage,
        UserCart $_cart,
        UsersFavContests $_userFav,
        Subscribers $_subscribers,
        OrdersTickets $_ordersTickets,
        Questions $_questions,
        UserAnswers $_userQuestion,
        EmailService $emailService
    ) {
        $this->contactUs    =   $_contact_us;
        $this->contactPage  =   $_contact_page;
        $this->aboutUs      =   $_about_us;
        $this->contests     =   $_contests;
        $this->homepage     =   $_homepage;
        $this->cart         =   $_cart;
        $this->fav          =   $_userFav;
        $this->subscribers  =   $_subscribers;
        $this->ordersTickets =   $_ordersTickets;
        $this->questions    =   $_questions;
        $this->userQuestion =   $_userQuestion;
        $this->emailService = $emailService;
    }

    public function index(Request $request)
    {
        $contests =
            $this->contests->where(function ($query) use ($request) {
                $query->whereDate('contest_end', '>', date('Y-m-d'));
                // $query->where(
                //     'qty',
                //     '!=',
                //     function ($subQuery) {
                //         $subQuery->from('orders_tickets')
                //             ->selectRaw('COUNT(*)')
                //             ->whereColumn('orders_tickets.contest_id', 'contests.id');
                //     }
                // );
            })
            ->where('featured', 1)
            ->orderByRaw('DATEDIFF(contests.contest_end, ?) ASC', [Carbon::now()->toDateString()])
            ->limit(6)
            ->get();

        $banner     =   $this->homepage->where('type', 'banner')->first();
        $howToPlay  =   $this->homepage->where('type', 'how_to_play')->first();
        $context    =   $this->homepage->where('type', 'context')->first();
        $feature    =   $this->homepage->where('type', 'feature')->first();
        $support    =   $this->homepage->where('type', 'support')->first();
        $supportTeam    =   $this->homepage->where('type', 'support_team')->first();
        $supportFaqs    =   $this->homepage->where('type', 'support_rfq')->first();
        return view('application.home', compact('contests', 'banner', 'howToPlay', 'context', 'feature', 'support', 'supportTeam', 'supportFaqs'));
    }

    public function aboutUs()
    {
        $testimonials = $this->aboutUs->where('type', 'testimonials')->orderBy('id', 'DESC')->get();
        $teamMembers = $this->aboutUs->where('type', 'team')->orderBy('id', 'DESC')->get();
        $aboutUsIntro  = $this->aboutUs->where('type', 'intro')->first();
        $aboutUsTestimonialIntro  = $this->aboutUs->where('type', 'testimonial_intro')->first();
        $aboutUsTeamIntro  = $this->aboutUs->where('type', 'team_intro')->first();
        $aboutUsWhatsDiffIntro  = $this->aboutUs->where('type', 'whats_different_intro')->first();
        return view('application.about-us', compact('testimonials', 'teamMembers', 'aboutUsIntro', 'aboutUsTestimonialIntro', 'aboutUsTeamIntro', 'aboutUsWhatsDiffIntro'));
    }

    public function contactUs()
    {
        $contact = $this->contactPage->where('type', 'contact_page_master_data')->first();
        $contactPhone = $this->contactPage->where('type', 'contact_page_phone')->first();
        $contactEmail = $this->contactPage->where('type', 'contact_page_email')->first();
        return view('application.contact-us', compact('contact', 'contactPhone', 'contactEmail'));
    }

    public function contactUsRequestSave(Request $request)
    {
        try {
            $response = DB::transaction(function () use ($request) {
                $data = $request->all();
                $contactUs = $this->contactUs->create($data);
                if ($contactUs) {
                    //Todo:: Mail functionality will place here.
                    $data = [
                        'subject'   => 'Contact Request',
                        'request'   => $request->all(),
                    ];
                    // Mail::to($contactUs->email)->send(new ContactRequestEmail($data));

                    return response()->json(['status' => true, 'message' => 'Thankyou, Your request as sent, Your team will connect you soon.'], 200);
                }

                return response()->json(['status' => true, 'message' => 'Thankyou, Your request as sent, Your team will connect you soon.'], 200);
            });

            return $response;
        } catch (\Throwable $th) {
            return response()->json(['status' => false, 'message' => $th->getMessage()], 200);
        }
    }

    public function faq()
    {

        $faqCategories = FAQCategories::get();
        return view('application.faq', compact('faqCategories'));
    }

    public function privacy(Request $request)
    {
        $privacy  =   $this->aboutUs->where('type', 'privacy')->first();
        return view('application.privacy', compact('privacy'));
    }

    public function termServices(Request $request)
    {
        $terms  =   $this->aboutUs->where('type', 'terms_services')->first();
        return view('application.terms_services', compact('terms'));
    }

    public function disclaimer(Request $request)
    {
        $disclaimer  =   $this->aboutUs->where('type', 'disclaimer')->first();
        return view('application.disclaimer', compact('disclaimer'));
    }

    // User Account Controller Methods
    public function dashboard(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('home');
        }
        $userId = Auth::user()->id;
        $timezones = \DateTimeZone::listIdentifiers();
        $user = User::with('additional_info')->find($userId);
        return view('application.user.dashboard', compact('user'));
    }

    public function tickets(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('home');
        }
        $user = Auth::user();
        $upComingTickets = OrdersTickets::select(
            'orders_tickets.id',
            'orders_tickets.ticket_id',
            'orders_tickets.ticket_no',
            'orders_tickets.ticket_status',
            'orders.created_at',
            'contests.title',
            'contests.contest_no',
            'contests.contest_on',
            'contests.contest_end',
            'contests.draw_date'
        )
            ->leftJoin('orders_items', 'orders_items.id', 'orders_tickets.item_id')
            ->leftJoin('orders', 'orders.id', 'orders_items.order_id')
            ->leftJoin('contests', 'contests.id', 'orders_items.contest_id')
            ->where('contests.deleted_at', null)
            ->where('orders.user_id', $user->id)
            ->whereDate('contests.draw_date', '>', date('Y-m-d'))
            ->get();

        $activeTickets = OrdersTickets::select(
            'orders_tickets.id',
            'orders_tickets.ticket_id',
            'orders_tickets.ticket_no',
            'orders_tickets.ticket_status',
            'orders.created_at',
            'contests.title',
            'contests.contest_no',
            'contests.contest_on',
            'contests.contest_end',
            'contests.draw_date'
        )
            ->leftJoin('orders_items', 'orders_items.id', 'orders_tickets.item_id')
            ->leftJoin('orders', 'orders.id', 'orders_items.order_id')
            ->leftJoin('contests', 'contests.id', 'orders_items.contest_id')
            ->where('contests.deleted_at', null)
            ->where('orders.user_id', $user->id)
            ->whereDate('contests.draw_date', '=', date('Y-m-d'))
            // ->whereDate('contests.contest_end', '<', date('Y-m-d'))
            ->get();

        $page = $request->input('page', 1);
        $perPage = 10;

        $pastTickets = OrdersTickets::select(
            'orders_tickets.id',
            'orders_tickets.ticket_id',
            'orders_tickets.ticket_no',
            'orders_tickets.ticket_status',
            'orders_tickets.updated_at as draw_updated_at',
            'orders.created_at',
            'contests.title',
            'contests.contest_no',
            'contests.contest_on',
            'contests.contest_end',
            'contests.draw_date'
        )
            ->leftJoin('orders_items', 'orders_items.id', 'orders_tickets.item_id')
            ->leftJoin('orders', 'orders.id', 'orders_items.order_id')
            ->leftJoin('contests', 'contests.id', 'orders_items.contest_id')
            ->where('contests.deleted_at', null)
            ->where('orders.user_id', $user->id)
            ->whereDate('contests.draw_date', '<', date('Y-m-d'));

        if ($request->has('date_range') && $request->date_range) {
            $dates = explode('-', $request->date_range);
            $startDate = isset($dates[0]) ? date('Y-m-d H:i:s', strtotime(trim($dates[0]))) : '';
            $endDate = isset($dates[1]) ? date('Y-m-d H:i:s', strtotime(trim($dates[1]))) : '';

            if ($startDate && $endDate) {
                $pastTickets->whereDate('created_at', '>=', $startDate)
                    ->whereDate('created_at', '<=', $endDate);
            } else if ($startDate) {
                $pastTickets->whereDate('created_at', '=', $startDate);
            }
        }

        $pastTickets = $pastTickets->orderBy('orders.id', 'desc')->paginate($perPage, ['*'], 'page', $page);

        if ($request->ajax()) {
            return response()->json([
                'html' => view('application.user.partials.tickets', compact('pastTickets'))->render(),
                'current_page' => $pastTickets->currentPage(),
                'last_page' => $pastTickets->lastPage(),
            ]);
        }


        return view('application.user.my_tickets', compact('user', 'activeTickets', 'pastTickets', 'upComingTickets'));
    }

    public function transactions(Request $request)
    {
        $user = Auth::user();
        $page = $request->input('page', 1);
        $perPage = 10;
        $query = Orders::join('orders_items', 'orders_items.order_id', 'orders.id')
            ->join('contests', 'contests.id', 'orders_items.contest_id')
            ->select(
                'orders.id as order_id',
                'orders.created_at as date',
                'orders.order_response as order_response',
                'orders.payment_response as payment_response',
                'contests.contest_no',
                'contests.id as contest_id',
                'contests.title as contest_title',
                'orders_items.total as amount',
                'orders_items.quantity as total_tickets',
                'orders.payment_status',
                'contests.draw_date as draw_date'
            )->where('user_id', $user->id);


        if ($request->has('date_range') && $request->date_range) {
            $dates = explode('-', $request->date_range);
            $startDate = isset($dates[0]) ? date('Y-m-d H:i:s', strtotime(trim($dates[0]))) : null;
            $endDate = isset($dates[1]) ? date('Y-m-d H:i:s', strtotime(trim($dates[1]))) : null;

            if ($startDate && $endDate) {
                $query->whereDate('orders.created_at', '>=', $startDate)
                    ->whereDate('orders.created_at', '<=', $endDate);
            } else if ($startDate) {
                $query->whereDate('orders.created_at', '=', $startDate);
            }
        }

        $transactions = $query->orderBy('orders.id', 'desc')->paginate($perPage, ['*'], 'page', $page);

        if ($request->ajax()) {
            return response()->json([
                'html' => view('application.user.partials.transactions', compact('transactions'))->render(),
                'current_page' => $transactions->currentPage(),
                'last_page' => $transactions->lastPage(),
            ]);
        }

        return view('application.user.transactions', compact('user', 'transactions'));
    }

    public function referralProgram(Request $request)
    {
        $user = Auth::user();
        return view('application.user.referral_program', compact('user'));
    }

    public function favoriteLottery(Request $request)
    {
        $user = Auth::user();
        $favorites  =   $user->favContests;
        return view('application.user.fav_lottery', compact('user', 'favorites'));
    }

    public function makeOrUnmakeFav(Request $request)
    {
        $fav    =   $this->fav->where(['user_id' => $request->user_id, 'contest_id' =>  $request->contest_id])->first();
        if ($fav) {

            if ($fav->delete()) {
                return response()->json(['status' => false, 'message' => 'Contest removed from your favorite list.'], 200);
            }

            return response()->json(['status' => true, 'message' => 'Contest not removed from your favorite list.'], 200);
        } else {
            $fav = $this->fav->create(['user_id' => $request->user_id, 'contest_id' =>  $request->contest_id]);
            if ($fav) {
                return response()->json(['status' => true, 'message' => 'Contest added to your favorite list.'], 200);
            }

            return response()->json(['status' => false, 'message' => 'Contest not added to your favorite list.'], 200);
        }
    }

    public function subscribers(Request $request)
    {
        $subscriber = $this->subscribers->where('email', $request->email)->first();
        if ($subscriber) {
            return $this->success('Email as already subscribed');
        }

        $subscribe  =   $this->subscribers->create(['email' =>  $request->email]);
        if ($subscribe) {

            //Mail Functionality.
            $this->sendSubscription($subscribe);
            return $this->success('Thankyou for subscribe with us.');
        }

        return $this->success('Sorry, subscribe request as failed. try it later.');
    }


    public function sendSubscription($subscribe)
    {
        $to = $subscribe->email;
        $subject = 'You have successfully subscribed to BMFCompetitions! 🎉';
        $view = 'emails.subscribe';
        $data['subscribe'] = $subscribe;
        $send = $this->emailService->sendEmail($to, $data, $view, $subject);
        return $send;
    }

    public function getQuestionToAnswer(Request $request)
    {
        $question = $this->questions->first();
        return view('application.components.question', compact('question'));
    }

    public function saveUserQuestionAnswer(Request $request)
    {
        $question = $this->questions->find($request->question);
        if (!$question) {
            return response()->json(['status' => false, 'message' => 'Invalid question to answer.'], 200);
        }

        if (empty($request->answer)) {
            return response()->json(['status' => false, 'message' => 'Please select your answer.'], 200);
        }

        $user = Auth::user();

        // Retrieve the current number of attempts from the session
        $attempts = session()->get("attempts_" . $question->id, 0);

        // Check if attempts are exhausted
        if ($attempts >= 4) {
            session()->put("attempts_" . $question->id, 0);
            return response()->json(['status' => false, 'message' => 'Try again!', 'remaining_attempts' => 4 - ($attempts + 1)], 200);
        }

        // Validate the user's answer
        $userAnswer = $request->answer;
        if ($userAnswer != $question->correct_answer_id) {
            session()->put("attempts_" . $question->id, $attempts + 1);
            return response()->json(['status' => false, 'message' => 'Sorry! You selected the wrong answer, please try again.']);
        }


        $response = DB::transaction(function () use ($question, $request, $user) {

            $checkQuestion = $this->userQuestion->where(['user_id' => $user->id, 'question_id' => $question->id])->first();
            if ($checkQuestion) {
                $checkQuestion->answer_id = $request->answer;
                $checkQuestion->save();
                session()->forget("attempts_" . $question->id);
                return response()->json(['status' => true, 'message' => 'Congratulations! You selected the right answer and are eligible for this competition.'], 200);
            }

            $userQuestion = $this->userQuestion->create([
                'user_id'       =>  $user->id,
                'question_id'   =>  $question->id,
                'answer_id'     =>  $request->answer,
            ]);

            if ($userQuestion) {
                session()->forget("attempts_" . $question->id);
                return response()->json(['status' => true, 'message' => 'Congratulations! You selected the right answer and are eligible for this competition.'], 200);
            }

            return response()->json(['status' => false, 'message' => 'Sorry, Unable to submit your question.'], 200);
        });

        return $response;
    }
}
