<?php

namespace App\Http\Controllers\Application;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Traits\Bmf as BMF;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use App\Models\User;
use App\Models\UserAdditionalInfo;
use Illuminate\Support\Facades\Route;
use App\Services\EmailService;
use App\Models\ResetPassword;
use App\Models\UserCart;
use Carbon\Carbon;

class AuthController extends Controller
{
    use BMF;

    protected $emailService;
    protected $user;
    protected $forgetPassword;
    protected $cart;

    public function __construct(
        User $_user,
        EmailService $emailService,
        ResetPassword $forgetPassword,
        UserCart $_cart
    ) {
        $this->user = $_user;
        $this->cart = $_cart;
        $this->emailService = $emailService;
        $this->forgetPassword = $forgetPassword;
    }

    public function saveRegisterUserDetails(Request $request)
    {
        try {
            if ($request->password !== $request->confirm_password) {
                return response()->json(['status' => false, 'message' => 'Password and Confirm Password does not match', 'url' => null], 200);
            }

            $checkEmailExists = $this->user->where('email', $request->email)->get()->count();
            if ($checkEmailExists) {
                return response()->json(['status' => false, 'message' => 'Email is already exists.', 'url' => null], 200);
            }

            if ($request->confirm_password !== $request->password) {
                return response()->json(['status' => false, 'message' => 'Password and Confirm Password Not Matching.', 'url' => null], 200);
            }

            $response = DB::transaction(function () use ($request) {
                $userData = [
                    'name'      =>  $request->first_name . ' ' . $request->last_name,
                    'first_name'      =>  $request->first_name,
                    'last_name'      =>  $request->last_name,
                    'email'     =>  $request->email,
                    'password'  =>  Hash::make($request->confirm_password),
                    'user_type' =>  3,
                    'mobile'    =>  '+' . $request->country_code . ' ' . $request->phone_number
                    // 'email_verified_at' =>  now()
                ];

                $user = $this->user->create($userData);
                if ($user) {
                    $user->verified_account_token = md5(now() . '|=|' . $user->id);
                    $user->save();

                    UserAdditionalInfo::create([
                        'user_id' => $user->id,
                        'gender' => $request->gender,
                        'dob' => !empty($request->dob) ? Carbon::createFromFormat('m/d/Y', $request->dob)->format('Y-m-d') : null,
                        'address' => $request->street_address,
                        'city' => $request->city,
                        'state' => $request->state,
                        'postal_code' => $request->postal_code,
                        'country' => $request->country,
                        'country_code' => $request->country_code,
                        'phone' => $request->phone_number,
                        'flat_no'   => $request->flat_no,
                        'landmark'   => $request->landmark,
                        'iso2'   => $request->iso2,
                    ]);

                    $verifyURL = route('frontend.register.verify.account', ['token' => $user->verified_account_token]);
                    $user->verify_url = $verifyURL;
                    $user->verify_sec_url = $verifyURL . '?email=' . $user->email;

                    $url = route('frontend.user.dashboard');
                    return ['status' => true, 'message' => 'You have successfully registered. We have sent an activation link to your registered email. Please check and activate your account.', 'redirect' => $url, 'user' => $user];
                }

                return ['status' => false, 'message' => 'Something went wrong.', 'redirect' => null];
            });

            if ($response['status']) {
                $user = $response['user'];
                // Auth::loginUsingId($user->id);
                //Todo::Send Email Notification to confirm email id.
                $status = $this->sendVerificationLink($user);
            }

            return response()->json($response, 200);
        } catch (\Throwable $th) {
            return response()->json(['status' => false, 'message' => $th->getMessage(), 'redirect' => null], 501);
        }
    }

    public function sendVerificationLink($user)
    {
        if (isset($user->email) && !empty($user->email)) {
            $to = $user->email;
            $subject =  'Verify Your Account - BMF Competitions';
            $view = 'emails.register-new-user-verify';
            $data['user']   = $user;
            $send = $this->emailService->sendEmail($to, $data, $view, $subject);
            Log::info($send);
            return $send;
        }
    }

    public function verifyAccount(Request $request)
    {
        $token = $request->token;
        $user = $this->user->where('verified_account_token', $token);
        if (isset($request->email)) {
            $user->where('email', $request->email);
        }
        $user = $user->first();
        if (!$user) {
            return redirect()->route('home')->with('failed', 'Invalid user ticket to verify account.');
        }

        if ($token !== $user->verified_account_token) {
            return redirect()->route('home')->with('failed', 'Invalid token to verify account.');
        }

        if (!empty($user->email_verified_at)) {
            return redirect()->route('home')->with('failed', 'Sorry, Your account is already verified, please login.');
        }

        $user->status = 1;
        $user->email_verified_at = now();
        $user->save();
        return redirect()->route('home')->with('success', 'Your account is verified successfully. Please login to your account.');
    }

    public function login(Request $request)
    {
        return view('application.login');
    }

    public function loginUserAccess(Request $request)
    {
        try {

            $checkEmailExists = $this->user->where('email', $request->email)->get()->count();
            if (!$checkEmailExists) {
                return response()->json(['status' => false, 'message' => 'No login account found.'], 200);
            }

            $url = (env('APP_ENV') == 'local') ? route('home') : (url()->previous() ?? route('home'));
            $route = isset($request->redirect_url) ? $request->redirect_url : route('home');
            $response = DB::transaction(function () use ($request, $route, $url) {
                $remember = $request->has('remember_me');
                if (Auth::attempt(['email' => $request->email, 'password' => $request->password, 'user_type' => 3], $remember)) {
                    $user = Auth::user();

                    if (empty($user->email_verified_at)) {
                        Auth::logout();
                        return ['status' => false, 'message' => 'Please verify account email id.', 'redirect' => null];
                    }

                    $ip_address = $request->ip();
                    $orders = $this->cart->where('user_id', $user->id)->orWhere('ip_address', $ip_address)->get()->count();
                    $redirectUrl = $orders ? $route : route('frontend.user.dashboard');

                    if (isset($request->remember_me) && $request->remember_me == 1) {
                    }

                    return ['status' => true, 'message' => 'You logged in successfully.', 'redirect' => $redirectUrl];
                    // . '?req=questions'
                }

                return ['status' => false, 'message' => 'Invalid login Email or Password.', 'redirect' => null];
            });

            return response()->json($response, 200);
        } catch (\Throwable $th) {
            return response()->json(['status' => false, 'message' => $th->getMessage()], 501);
        }
    }

    public function logout()
    {
        Auth::logout();
        return redirect()->route('home');
    }

    public function forgetPassword(Request $request) {}

    public function forgetPasswordRequestStore(Request $request)
    {
        try {
            $user = $this->user->where('email', $request->email)->first();
            if (!$user) {
                return response()->json(['status' => false, 'message' => 'No account details found.'], 200);
            }

            $response = DB::transaction(function () use ($request, $user) {

                $forgetPassword = $this->forgetPassword->where('email', $user->email)->first();
                if ($forgetPassword) {
                    $forgetPassword->token = md5(now() . '|+|' . $user->id);
                    $forgetPassword->created_at = now();
                    $forgetPassword->updated_at = null;
                    $forgetPassword->save();
                } else {
                    $forgetPassword = $this->forgetPassword->create([
                        'email' => $user->email,
                        'token' => md5(now() . '|+|' . $user->id),
                        'created_at'    => now()
                    ]);
                }

                if ($forgetPassword) {

                    $link = route('frontend.forget-password.verify.token', ['token' => $forgetPassword->token]);
                    $user->reset_link = $link;
                    return [
                        'status'    =>  true,
                        'message'   =>  'Reset Password link as sent to email.',
                        'redirect'  =>  route('home'),
                        'link'      =>  $link,
                        'forgetPassword' => $forgetPassword
                    ];
                }

                return ['status' => false, 'message' => 'Failed to request reset password.', 'redirect' => null];
            });

            if ($response['status']) {
                // Send Email
                if (isset($user->email) && !empty($user->email)) {
                    $to = $user->email;
                    $subject =  'Password Reset Request';
                    $view = 'emails.forget-password-request';
                    $data['user']   = $user;
                    $data['reset_link']   = $response['link'];
                    $data['forgetPassword']   = $response['forgetPassword'];
                    $send = $this->emailService->sendEmail($to, $data, $view, $subject);
                    Log::info($send);
                }
            }

            return response()->json($response, 200);
        } catch (\Throwable $th) {
            return response()->json(['status' => false, 'message' => $th->getMessage()], 501);
        }
    }

    public function forgetPasswordVerifyToken(Request $request)
    {
        $token = $request->token;
        $forgetPassword = $this->forgetPassword->where('token', $token)->where('updated_at', null)->first();
        if (isset($forgetPassword->token) && $forgetPassword->token === $token) {
            return view('application.reset_password', compact('forgetPassword'));
        } else {
            return redirect()->route('frontend.login')->with('failed', 'Reset Password token as invalid.');
        }
    }

    public function forgetedUpdateNewPassword(Request $request)
    {
        try {
            $user = $this->user->where('email', $request->email)->first();
            if (!$user) {
                return redirect()->route('frontend.login')->with('failed', 'No account details found.');
            }

            $forgetPassword = $this->forgetPassword->where('token', $request->token)->first();
            $response = DB::transaction(function () use ($request, $user, $forgetPassword) {

                if ($request->password !== $request->confirm_password) {
                    return ['status' => false, 'message' => 'Password and confirm password should be same.'];
                }

                $user->password = Hash::make($request->password);
                if ($user->save()) {
                    $forgetPassword->updated_at = now();
                    $forgetPassword->save();

                    return ['status' => true, 'message' => 'Password as updated successfully.'];
                }


                return ['status' => false, 'message' => 'Failed to update password.'];
            });

            if ($response['status']) {
                return redirect()->route('home')->with('success', $response['message']);
            }

            return redirect()->back()->with('failed',  $response['message']);
        } catch (\Throwable $th) {
            return redirect()->back()->with('failed',  $th->getMessage());
        }
    }
}
