<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Contests;
use App\Models\OrdersTickets;
use Illuminate\Support\Carbon;
use App\Services\EmailService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TicketsController extends Controller
{

    protected $emailService;

    /**
     * Create a new controller instance.
     *
     * @param EmailService $emailService
     * @return void
     */
    public function __construct(EmailService $emailService)
    {
        $this->emailService = $emailService;
    }

    public function index()
    {

        $tickets = OrdersTickets::with('contest')
            ->select(
                DB::raw('count(ticket_id) as ticket_count'),
                'contest_id',
                'orders_tickets.id'
            )
            ->groupBy('orders_tickets.contest_id')
            ->get();
        return view('admin.tickets.list', compact('tickets'));
    }

    public function declareWinner(Request $request)
    {

        try {
            DB::beginTransaction();
            $ticketNumber = $request->input('ticket_number');
            $contestId = $request->input('contest_id');
            $checkExistency = OrdersTickets::where(['contest_id' => $contestId, 'ticket_id' => $ticketNumber])->first();

            if (!$checkExistency) {
                return response()->json(['message' => 'Ticket number is not matching', 'status' => 2]);
            } else if ($checkExistency->ticket_status == 'open') {

                $contest = Contests::find($contestId);
                if (Carbon::parse($contest->draw_date)->isPast()) {
                    return response()->json(['message' => 'You Can\'t declare winner yet or Draw date is over', 'status' => 2]);
                } else {

                    if (Carbon::parse($contest->draw_date)->isToday()) {
                        $checkExistency->ticket_status = 'win';
                        $checkExistency->save();

                        if ($checkExistency) {

                            $sendMail = $this->sendMailToWinner($checkExistency, $contest);
                        }

                        DB::commit();

                        return response()->json(['message' => 'Winner declared successfully', 'status' => 1]);
                    } else {
                        return response()->json(['message' => 'You can declare the winner on the Draw date only', 'status' => 2]);
                    }
                }
            } else {
                return response()->json(['message' => 'Something went wrong, please try again', 'status' => 2]);
            }
        } catch (\Exception $e) {
            Log::info(print_r($e->getMessage(), true) . ' error msg');
            DB::rollback();
        }
    }

    public function sendMailToWinner($input, $contest)
    {
        $to = $input->customer->email;
        $subject = 'Congratulations, You\'re a Winner! 🎉';
        $view = 'emails.winner-declare';
        $data['ticketDetails'] = $input;
        $send = $this->emailService->sendEmail($to, $data, $view, $subject, null, 'bmfcompetitions@gmail.com');
        $consolationMail = $this->sendConsolationMail($input, $contest);
        return $send;
    }

    public function sendConsolationMail($input, $contest)
    {
        $winnerId = $input->user_id;
        $contestId = $input->contest_id;
        $orderContest = OrdersTickets::where('contest_id', '=', $contestId)
            ->where('user_id', '<>', $winnerId)
            ->groupBy('user_id')
            ->get();
        $customerIds = $orderContest->pluck('user_id')->toArray();
        $customers = DB::table('users')->whereIn('id', $customerIds)->where(['user_type' => 3, 'status' => 1])->whereNull('deleted_at')->get();
        $currentDateTime = Carbon::now();
        $otherContests = Contests::where('draw_date', '>', $currentDateTime)->where('id', '<>', $contestId)->get();
        foreach ($customers as $customer) {
            $to = $customer->email;
            $subject = 'Thank You for Participating!';
            $view = 'emails.consolation-mail';
            $data['other_contests'] = $otherContests;
            $data['customer'] = $customer;
            $send = $this->emailService->sendEmail($to, $data, $view, $subject);
        }
        return true;
    }

    public function viewMoreTicketDetails()
    {


        $contestId = request()->contest_id;
        $contest = Contests::select('*')
            ->where('id', $contestId)
            ->first();

        return view('admin.tickets.view-more', compact('contest'))->render();
    }
}
