<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\News;
use Carbon\Carbon;
use App\Models\Category;
use Illuminate\Support\Facades\Storage;


class NewsController extends Controller
{
    public function index()
    {
        $news = News::select('id', 'title', 'category_id', 'news_date', 'image')
            ->with('category') // Ensure category relationship is loaded
            ->orderBy('id', 'desc') // Latest news first
            ->paginate(10); // 10 news per page

        return view('admin.news.list', compact('news'));
    }

    public function add()
    {
        $categories = Category::all();
        return view('admin.news.create',compact('categories'));
    }

    public function createNews(Request $request)
    {
        $validatedData = $request->validate([
            'category_id' => 'required|exists:categories,id',
            'title' => 'required|string|max:255',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'short_description' => 'required',
            'long_description' => 'required',
            'news_date' => 'required|date'
        ]);

        $Data = $request->except('_token');

        if (isset($Data['image'])) {
            $Icon = $Data['image'];
            $name = time() . '.' . $Icon->getClientOriginalExtension();
            $path = public_path('assets/news');
            $Icon->move($path, $name);
            $Data['image'] = $name;
        }
        $news = new News();
        $Save = $news->AddNewsDetails($Data);

        if ($Save) {
            return redirect()->route('admin.news')->with('success', 'News saved successfully.');

        } else {
            return redirect()->route('AddNews')->with('error', 'Something went wrong.');
        }
    }

    public function edit($id)
    {
        $news = News::find($id);
        $categories = Category::all();
        return view('admin.news.edit', compact('news', 'categories'));

    }

    public function update(Request $request, $id)
    {    
        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'image' => 'nullable|image|mimes:jpeg,jpg,gif|max:2048', // Changed to nullable
            'short_description' => 'required',
            'long_description' => 'required',
            'news_date' => 'required|date'
        ]);
        
        try {
            $news = News::findOrFail($id);
            $data = request()->except(['removed_images', '_token']);
            if (isset($data['image'])) {
                // Delete the old image if it exists
                $path = realpath(public_path('assets/news/' . $news->image));
                if ($news->image && file_exists($path)) {
                    unlink($path); 
                }
                // Upload new image
                $image = $data['image'];
                $imageName = time() . '.' . $image->getClientOriginalExtension();
                $image->move(public_path('assets/news'), $imageName);
                $data['image'] = $imageName;
            } else {
                // Keep the old image if no new one is uploaded
                $data['image'] = $news->image;
            }

            // Delete removed images from storage
            if ($request->has('removed_images')) {
                $removedImages = json_decode($request->removed_images, true);
                if (!empty($removedImages)) {
                    foreach ($removedImages as $imageUrl) {

                        $imagePath = ltrim(parse_url($imageUrl, PHP_URL_PATH), '/');
                        $imagePath = public_path(str_replace('dovehouse/public/', '', $imagePath));

                        // $imagePath = public_path(str_replace('dovehouse-Capital/public/', '', $imagePath));

                        if (file_exists($imagePath)) {
                            unlink($imagePath); // Delete image from storage
                        }
                    }
                }
            }

            // Update news record
            $news->update($data);

            return redirect()->route('admin.news')->with('success', 'News updated successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }
    

    public function delete($id)
    {

        if ($id) {
            $news = News::find($id);
            $delete = $news->delete();

            if ($delete) {
                return redirect()->route('admin.news')->with('success', 'News deleted successfully.');

            } else {
                return redirect()->route('admin.news')->with('error', 'news not deleted');
            }
        }
    }
    
public function uploadImage(Request $request)
{
    // Validate the request
    $validator = Validator::make($request->all(), [
        'upload' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048'
    ]);

    if ($validator->fails()) {
        return response()->json([
            'uploaded' => 0,
            'error' => ['message' => 'Cannot upload. File size must be less than 2MB.']
        ]);
    }
    
    if ($request->hasFile('upload')) {
        $file = $request->file('upload');

        // Generate a clean and unique filename
        $originalName = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $extension = $file->getClientOriginalExtension();
        $cleanName = preg_replace('/[^A-Za-z0-9_-]/', '_', $originalName);
        $filename = time() . '_' . $cleanName . '.' . $extension;

        // Store file in the public/uploads/ckeditor directory
        $file->move(public_path('uploads/ckeditor'), $filename);

        // Generate the correct URL for accessing the image
        $imageUrl = url('uploads/ckeditor/' . $filename);

        return response()->json([
            'uploaded' => 1,
            'fileName' => $filename,
            'url' => $imageUrl
        ]);
    }

    return response()->json([
        'uploaded' => 0,
        'error' => ['message' => 'No file was uploaded.']
    ]);
}




}
