<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use App\Http\Requests\ContestFormRequest;
use App\Http\Requests\ContestUpdateFormRequest;
use App\Models\Contests;
use App\Http\Traits\Bmf;
use Illuminate\Support\Facades\Storage;
use App\Models\UsersFavContests;
use App\Services\EmailService;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ContestController extends Controller
{
    use Bmf;

    protected $contestRepository;
    protected $contestImageRepository;
    protected $emailService;

    /**
     * Create a new controller instance.
     *
     * @param EmailService $emailService
     * @return void
     */
    public function __construct(
        \App\Repositories\ContestRepository $contestRepository,
        \App\Repositories\ContestImageRepository $contestImageRepository,
        EmailService $emailService
    ) {
        $this->contestRepository = $contestRepository;
        $this->contestImageRepository = $contestImageRepository;
        $this->emailService = $emailService;
        //$this->middleware('role:admin'); // Restrict access to admins only
    }

    public function index()
    {
        $contests = Contests::orderBy('id', 'desc')->get();
        return view('admin.contests.list', compact('contests'));
    }

    public function create()
    {

        return view('admin.contests.create');
    }

    public function store(ContestFormRequest $request)
    {
        // $customers = DB::table('users')->where('user_type', 3)->whereNull('deleted_at')->pluck('email');
        // $subscribers = DB::table('subscribers')->where('is_subscribe', 1)->pluck('email');
        // $uniqueEmails = $customers->merge($subscribers)->unique();

        $input = $request->all();
        $input['slug'] = Str::slug($input['title'], '-');
        $input['contest_number'] = $input['contest_no'];
        $input['featured'] = isset($input['featured']) ? 1 : 0;
        $contest = $this->contestRepository->create($input);
        if ($contest && $request->hasFile('images')) {
            foreach ($request->images as $image) {
                $imgData = $this->uploadWithFile($image, 'uploads/contests');
                $imageRepository = $this->contestImageRepository->create(['path' => $imgData['path'], 'contest_id' => $contest->id]);
            }
        }

        $this->sendContestDetails($contest);

        session()->flash('success', 'Contest has been created successfully!');
        return redirect()->route('admin.contests')->with('success', 'Contest created successfully!');
    }

    public function sendContestDetails($contest)
    {
        $customers = DB::table('users')->where('user_type', 3)->whereNull('deleted_at')->get();
        $currentDateTime = Carbon::now();
        $otherContests = Contests::where('draw_date', '>', $currentDateTime)->where('id', '<>', $contest->id)->get();
        foreach ($customers as $customer) {
            $to = $customer->email;
            $subject = 'Exciting New Competition Alert! 🎉';
            $view = 'emails.new-contest';
            $data['contest'] = $contest;
            $data['customer'] = $customer;
            $data['other_contests'] = $otherContests;
            $send = $this->emailService->sendEmail($to, $data, $view, $subject);
        }

        return true;
    }

    public function delete($id)
    {

        $delete = $this->contestRepository->find($id)->delete();

        if ($delete) {
            UsersFavContests::where('contest_id', $id)->delete();
            session()->flash('success', 'Contest deleted successfully!');
        } else {
            session()->flash('success', 'Contest not deleted, please try again!');
        }

        return redirect()->route('admin.contests');
    }

    public function imageDelete(Request $request)
    {
        $request = $request->all();
        $image = $this->contestImageRepository->find($request['galleryId']);
        if ($image && Storage::exists($image->path)) {
            $id = $request['galleryId'];
            $delete = $this->contestImageRepository->find($id)->delete();
            if ($delete) {
                Storage::delete($image->path);
            }

            $msg = 'Image has been deleted successfully!';
            $flag = true;
            //session()->flash('success', $msg);

        } else {
            $msg = 'Image not deleted, please try again!';
            $flag = false;
            //session()->flash('success', $msg);

        }

        return  response()->json(['status' => $flag, 'message' => $msg], 200);
    }

    public function update($id)
    {
        $contest = $this->contestRepository->find($id);
        return view('admin.contests.edit', compact('contest'));
    }

    public function updateContest(ContestUpdateFormRequest $request)
    {

        $input = $request->all();
        $input['slug'] = Str::slug($input['title'], '-');
        $input['contest_number'] = $input['contest_no'];
        $input['featured'] = isset($input['featured']) && $input['featured'] == 'on' ? 1 : 0;
        unset($input['id']);
        unset($input['_token']);
        unset($input['images']);
        $contest = $this->contestRepository->where('id', $request->id)->update($input);

        if ($contest && $request->hasFile('images')) {
            foreach ($request->images as $image) {
                $imgData = $this->uploadWithFile($image, 'uploads/contests');
                $imageRepository = $this->contestImageRepository->create(['path' => $imgData['path'], 'contest_id' => $request->id]);
            }
        }
        session()->flash('success', 'Contest has been updated successfully!');

        return redirect()->route('admin.contests')->with('success', 'Contest updated successfully!');
    }

    public function featured()
    {

        $contests = Contests::where('featured', 1)->orderBy('sort', 'asc')->paginate(10);
        return view('admin.contests.featured-list', compact('contests'));
    }

    public function editFeatured($id)
    {
        $contest = $this->contestRepository->find($id);
        return view('admin.contests.update-featured-contest', compact('contest'));
    }

    public function updateFeaturedContest(Request $request)
    {

        $input = $request->all();
        unset($input['_token']);
        $input['featured'] = isset($input['featured']) && $input['featured'] == 'on' ? 1 : 0;
        $contest = $this->contestRepository->where('id', $request->id)->update($input);

        session()->flash('success', 'Featured Contest has been updated successfully!');

        return redirect()->route('admin.contests')->with('success', 'Featured Contest updated successfully!');
    }
}
